#!/bin/bash
# Cellframe Node CLI Installation and Testing Script
# For integration into node build pipeline

set -e  # Exit on any error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
NODE_PACKAGE=${1:-"cellframe-node-latest.deb"}
NODE_URL="http://127.0.0.1:12345/connect"
WAIT_TIMEOUT=60
TEST_OUTPUT_DIR="test_results"

echo -e "${BLUE}🚀 Cellframe Node CLI Testing Pipeline${NC}"
echo "================================================"

# Function to log messages
log_info() {
    echo -e "${BLUE}ℹ️ $1${NC}"
}

log_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

log_warning() {
    echo -e "${YELLOW}⚠️ $1${NC}"
}

log_error() {
    echo -e "${RED}❌ $1${NC}"
}

# Check if running as root for node installation
check_root() {
    if [[ $EUID -eq 0 ]]; then
        log_info "Running as root - can install node package"
        return 0
    else
        log_warning "Not running as root - assuming node is already installed"
        return 1
    fi
}

# Install node from package if provided and running as root
install_node() {
    if [ -f "$NODE_PACKAGE" ] && check_root; then
        log_info "Installing Cellframe Node from $NODE_PACKAGE"
        
        # Stop any existing node service
        systemctl stop cellframe-node 2>/dev/null || true
        
        # Install the package
        dpkg -i "$NODE_PACKAGE" || {
            log_error "Failed to install node package"
            return 1
        }
        
        # Configure node for testing (minimal config)
        cat > /opt/cellframe-node/etc/cellframe-node.cfg << EOF
[general]
node_role=light
debug_mode=true

[server]
enabled=true
listen_address=127.0.0.1
listen_port=12345

[logging]
log_level=INFO
EOF
        
        log_success "Node package installed"
        return 0
    else
        log_info "Skipping node installation (package not found or not root)"
        return 0
    fi
}

# Start node service
start_node() {
    log_info "Starting Cellframe Node service"
    
    if check_root; then
        systemctl start cellframe-node
        systemctl enable cellframe-node
    else
        # Try to start manually if not root
        /opt/cellframe-node/bin/cellframe-node -D || {
            log_warning "Could not start node automatically"
        }
    fi
}

# Wait for node to be ready
wait_for_node() {
    log_info "Waiting for node to become ready..."
    
    local count=0
    while [ $count -lt $WAIT_TIMEOUT ]; do
        if curl -s "$NODE_URL" > /dev/null 2>&1; then
            log_success "Node is ready and responding"
            return 0
        fi
        
        echo -n "."
        sleep 1
        ((count++))
    done
    
    log_error "Node did not become ready within $WAIT_TIMEOUT seconds"
    return 1
}

# Install Python dependencies
install_dependencies() {
    log_info "Installing Python dependencies"
    
    # Create virtual environment if it doesn't exist
    if [ ! -d "venv" ]; then
        python3 -m venv venv
    fi
    
    # Activate virtual environment and install dependencies
    source venv/bin/activate
    pip install -r requirements.txt
    
    log_success "Dependencies installed"
}

# Run CLI tests
run_tests() {
    log_info "Running API validation tests"
    
    # Activate virtual environment
    source venv/bin/activate
    
    # Create output directory
    mkdir -p "$TEST_OUTPUT_DIR"
    
    # Run the tests
    python3 test_node.py --node-url "$NODE_URL" --output-dir "$TEST_OUTPUT_DIR"
    local test_exit_code=$?
    
    # Display results
    if [ $test_exit_code -eq 0 ]; then
        log_success "All tests passed! ✅"
    else
        log_error "Some tests failed! ❌"
    fi
    
    # Show latest report if exists
    local latest_report=$(ls -t "$TEST_OUTPUT_DIR"/test_report_*.json 2>/dev/null | head -1)
    if [ -n "$latest_report" ]; then
        log_info "Latest test report: $latest_report"
        
        # Extract and display summary
        if command -v jq >/dev/null 2>&1; then
            echo ""
            echo "📊 Quick Summary:"
            jq -r '.summary | "Total: \(.total_tests), Passed: \(.passed), Failed: \(.failed), Errors: \(.errors), Success Rate: \(.success_rate)"' "$latest_report"
        fi
    fi
    
    return $test_exit_code
}

# Cleanup function
cleanup() {
    log_info "Cleaning up..."
    
    if check_root; then
        systemctl stop cellframe-node 2>/dev/null || true
    else
        pkill -f cellframe-node 2>/dev/null || true
    fi
}

# Main execution flow
main() {
    # Set trap for cleanup on exit
    trap cleanup EXIT
    
    # Step 1: Install node if package provided
    install_node
    
    # Step 2: Install Python dependencies
    install_dependencies
    
    # Step 3: Start node
    start_node
    
    # Step 4: Wait for node to be ready
    wait_for_node
    
    # Step 5: Run tests
    run_tests
    local final_exit_code=$?
    
    echo ""
    echo "================================================"
    if [ $final_exit_code -eq 0 ]; then
        log_success "🎉 All validation tests completed successfully!"
        echo -e "${GREEN}Node API is backward compatible and working correctly.${NC}"
    else
        log_error "💥 Validation tests failed!"
        echo -e "${RED}Node API has compatibility issues or errors.${NC}"
    fi
    echo "================================================"
    
    exit $final_exit_code
}

# Help function
show_help() {
    echo "Usage: $0 [NODE_PACKAGE]"
    echo ""
    echo "Install Cellframe Node and run API validation tests"
    echo ""
    echo "Arguments:"
    echo "  NODE_PACKAGE    Path to .deb package file (optional)"
    echo ""
    echo "Environment variables:"
    echo "  NODE_URL        Node URL to test (default: $NODE_URL)"
    echo "  WAIT_TIMEOUT    Seconds to wait for node (default: $WAIT_TIMEOUT)"
    echo ""
    echo "Examples:"
    echo "  $0                              # Test existing node installation"
    echo "  $0 cellframe-node-latest.deb   # Install package and test"
    echo ""
}

# Handle command line arguments
case "${1:-}" in
    -h|--help)
        show_help
        exit 0
        ;;
    *)
        main "$@"
        ;;
esac
